###############################################################################%
# This code is to generate the Taylor Rule figures for the U.S.
###############################################################################%

### ======= Load data ======= ###
rm(list = ls())
df_q <- read_csv('Data/data_us_quarterly.csv')



###############################################################################%
#### Figure 2 - Fit of Original Taylor Rule, 1987-1992 ####
###############################################################################%


aux <- df_q %>% 
  filter(
    date >= '1984-01-01',
    date <= '1992-07-01'
  ) %>%
  # Follow Taylor compute output gap as deviation from trend real GDP
  # from the period 1984:Q1 to 1992:Q3, using Taylor's vintage
  mutate(
    `trend gdp` = lm(real_gdp_v93 ~ date, data = .)$fitted.values,
    y =  100 * (real_gdp_v93 / `trend gdp` - 1),
    p = inflation_def_v93,
    `Taylor Rule` = 2 + p + 0.5 * (p - 2) + 0.5 * y
  ) %>% 
  select(date, y, p, `Taylor Rule`, `Fed Funds Rate`)


aux %>% 
  filter(date >= '1987-01-01') %>%
  pivot_longer(cols = c(`Taylor Rule`, `Fed Funds Rate`)) %>% 
  ggplot(aes(date, value, color = name)) +
  geom_line() +
  labs(y = "Percent") +
  scale_color_manual(values = c("#355070", "#E56B6F")) 
ggsave('Output/fig2.pdf', height = 2.5, width = 2.5)


###############################################################################%
#### Figure 3 - Original Taylor Rule with Real-Time Data ####
###############################################################################%

aux <- df_q %>% 
  mutate(
    # Use CBO output gap when Greenbook not available
    y = case_when(
      !is.na(gap_gb_rt) ~ gap_gb_rt,
      TRUE ~ gap_cbo_rt
    ),
    p = deflator_real_time,
    `Taylor Rule` = 2 + p + 0.5 * (p - 2) + 0.5 * y
  ) %>% 
  select(date, y, p, `Taylor Rule`, `Fed Funds Rate`)


aux %>% 
  pivot_longer(
    cols = c(`Taylor Rule`, `Fed Funds Rate`)
  ) %>% 
  ggplot(aes(date, value, color = name)) +
  geom_line() +
  scale_color_manual(values = c("#355070", "#E56B6F")) +
  labs(y = 'Percent') +
  annotate("rect", xmin = as.Date("1987-01-01"), xmax = as.Date("1992-09-01"), 
           ymin = -Inf, ymax = Inf, fill = "darkgrey", alpha = 0.4) +
  geom_hline(yintercept = 0, linewidth=0.3) +
  scale_x_date(breaks = as.Date(paste0(seq(1960, 2020, by = 10), "-01-01")),
               date_labels = "%Y")
ggsave('Output/fig3.pdf', height = 2.6, width = 6)



###############################################################################%
#### Figure 4 - Original Taylor Rule with Retrospective Data ####
###############################################################################%

aux <- df_q %>% 
  mutate(
    # Use CBO output gap when Greenbook not available
    y = case_when(
      !is.na(gap_gb_final)  ~ gap_gb_final,
      TRUE ~ gap_cbo_final
    ),
    p = deflator_final_vintage,
    `Taylor Rule` = 2 + p + 0.5 * (p - 2) + 0.5 * y
  ) %>% 
  select(date, y, p, `Taylor Rule`, `Fed Funds Rate`)


aux %>% 
  pivot_longer(
    cols = c(`Taylor Rule`, `Fed Funds Rate`)
  ) %>% 
  ggplot(aes(date, value, color = name)) +
  geom_line() +
  scale_color_manual(values = c("#355070", "#E56B6F")) +
  labs(y = 'Percent') +
  annotate("rect", xmin = as.Date("1987-01-01"), xmax = as.Date("1992-09-01"), 
           ymin = -Inf, ymax = Inf, fill = "darkgrey", alpha = 0.4) +
  geom_hline(yintercept = 0, linewidth=0.3) +
  scale_x_date(breaks = as.Date(paste0(seq(1960, 2020, by = 10), "-01-01")),
               date_labels = "%Y")
ggsave('Output/fig4.pdf', height = 2.6, width = 6)



###############################################################################%
#### Figure 6 - Clarida, Gali, Gertler Policy Rule with Real-Time Data ####
###############################################################################%
aux <- df_q %>% 
  mutate(
    # Use CBO output gap when Greenbook not available
    y = case_when(
      !is.na(gap_gb_rt) ~ gap_gb_rt,
      TRUE ~ gap_cbo_rt
    ),
    y = lead(y, 1),
    # Use the SPF forecast when the Greenbook forecast is not available
    p = case_when(
      !is.na(p_gb_rt1) ~ p_gb_rt1,
      TRUE ~ p_spf_rt1
    ),
    `Policy Rule` =  4.11 - (2.15 - 1) * 3.58 +
      2.15 * p + 0.93 * y,
  )  %>% 
  select(date, y, p, `Policy Rule`, `Fed Funds Rate`)



aux %>% 
  pivot_longer(
    cols = c(`Policy Rule`, `Fed Funds Rate`)
  ) %>% 
  ggplot(aes(date, value, color = name)) +
  geom_line() +
  scale_color_manual(values = c("#355070", "#E56B6F")) +
  geom_hline(yintercept = 0, linewidth=0.3) +
  labs(y = 'Percent') +
  annotate("rect", xmin = as.Date("1987-01-01"), xmax = as.Date("1992-09-01"), 
           ymin = -Inf, ymax = Inf, fill = "darkgrey", alpha = 0.4) +
  scale_x_date(breaks = as.Date(paste0(seq(1960, 2020, by = 10), "-01-01")),
               date_labels = "%Y")
ggsave('Output/fig6.pdf', height = 2.6, width = 6)



###############################################################################%
#### Figure 7 - Balanced-Approach Rule with Core-PCE Inflation, and Time-Varying R∗ ####
###############################################################################%

aux <- df_q %>% 
  mutate(
    # Use CBO output gap when Greenbook not available
    y = case_when(
      !is.na(gap_gb_rt) ~ gap_gb_rt,
      TRUE ~ gap_cbo_rt
    ),
    # Use retrospective pce inflation when real-time not available
    p = case_when(
      is.na(rt_pce_core) ~ inflation_pce_core,
      !is.na(rt_pce_core) ~ rt_pce_core
    ),
    # Suppose r* = 2 when the SEP rate is not available
    r_star = case_when(
      is.na(sep_rate) ~ 2,
      !is.na(sep_rate) ~ sep_rate - 2
    ),
    `Policy Rule` = r_star + p + 0.5 * (p - 2) + y
  ) %>% 
  select(date, y, p, r_star, `Policy Rule`, `Fed Funds Rate`)


aux %>% 
  pivot_longer(
    cols = c(`Policy Rule`, `Fed Funds Rate`)
  ) %>% 
  ggplot(aes(date, value, color = name)) +
  geom_line() +
  scale_color_manual(values = c("#355070", "#E56B6F")) +
  geom_hline(yintercept = 0, linewidth=0.3) +
  labs(y = 'Percent') +
  annotate("rect", xmin = as.Date("1987-01-01"), xmax = as.Date("1992-09-01"), 
           ymin = -Inf, ymax = Inf, fill = "darkgrey", alpha = 0.4) +
  scale_x_date(breaks = as.Date(paste0(seq(1960, 2020, by = 10), "-01-01")),
               date_labels = "%Y")
ggsave('Output/fig7.pdf', height = 2.6, width = 6)



###############################################################################%
#### Figure 8 - First Difference Policy Rule ####
###############################################################################%

# n = SPF median 3-quarter ahead forecast of nominal GDP growth 
# n* = (pi* = 2) + SPF median 10-year forecast for real GDP growth

### ======= Panel 1: true lagged values ======= ###
df_q %>% 
  drop_na(g_interp) %>% 
  mutate(
    n_star = 2 + g_interp,
    delta_i = (1/2) * (n - n_star), 
    # Compute policy rule as past rate + prescribed change
    level = lag(`Fed Funds Rate`, 1) + delta_i
  ) %>% 
  pivot_longer(
    cols = c(`Fed Funds Rate`, level)
  ) %>% 
  ggplot(aes(date, value, color = name)) +
  geom_line() +
  labs(y = 'Percent') +
  scale_color_manual(values = c("#355070", "#E56B6F"),
                     labels = c("Fed Funds Rate", 'Policy Rule')) +
  theme(
    axis.title.y = element_blank(),
    legend.position = c(0.36, 0.16),
    legend.margin = margin(t = 2, r = 4, b = 2, l = 4),  
    legend.justification = c("right", "top"),  
    legend.background = element_rect(fill = "white", color = "grey80"),
    legend.title = element_blank()
  ) +
  guides(color = guide_legend(nrow = 2))
ggsave('Output/fig8_panel1.pdf', height = 4, width = 4)



### ======= Panel 2: cumulative prescribed changes ======= ###

# Compute policy rule as cumulative prescribed changes
df_cumu <- df_q %>% 
  drop_na(g_interp) %>%
  mutate(
    n_star = 2 + g_interp,
    delta_i = (1/2) * (n - n_star),
    level = NaN
  )

df_cumu$level[1] <- df_cumu$`Fed Funds Rate`[1]

for (t in 2:nrow(df_cumu)) {
  df_cumu$level[t] <- df_cumu$level[t-1] + df_cumu$delta_i[t]
}

df_cumu %>% 
  pivot_longer(
    cols = c(`Fed Funds Rate`, level)
  ) %>% 
  ggplot(aes(date, value, color = name)) +
  geom_line() +
  labs(y = 'Percent') +
  scale_color_manual(values = c("#355070", "#E56B6F"),
                     labels = c("Fed Funds Rate", 'Policy Rule')) +
  theme(
    axis.title.y = element_blank(),
    legend.position = c(0.35, 0.16),
    legend.margin = margin(t = 2, r = 4, b = 2, l = 4), 
    legend.justification = c("right", "top"), 
    legend.background = element_rect(fill = "white", color = "grey80"),
    legend.title = element_blank()
  ) +
  guides(color = guide_legend(nrow = 2))
ggsave('Output/fig8_panel2.pdf', height = 4, width = 4)



###############################################################################%
#### Figure B3 - First Difference Policy Rule: Actual and Prescribed Change in Interest Rates####
###############################################################################%

df_q %>% 
  drop_na(g_interp) %>% 
  mutate(
    n_star = 2 + g_interp,
    delta_i = (1/2) * (n - n_star),
    delta_ffr = `Fed Funds Rate` - lag(`Fed Funds Rate`, 1)
  ) %>%  
  pivot_longer(
    cols = c(delta_ffr, delta_i)
  ) %>%
  ggplot(aes(date, value, color = name)) +
  geom_line() +
  labs(y = 'Percent') +
  scale_color_manual(values = c("#355070", "#E56B6F"),
                     labels = c("Fed Funds Rate", 'Policy Rule')) +
  theme(
    axis.title.y = element_blank(),
  ) 
ggsave('Output/figB3.pdf', height =  2.6, width = 6)
