###############################################################################%
# This code is to generate the G7 analysis pictures
###############################################################################%

### ======= Load data ======= ###
rm(list = ls())
df <- read_csv('Data/data_G7.csv')


###############################################################################%
#### Figure 9 - Taylor Rules for G7 Countries ####
###############################################################################%

df %>% 
  group_by(country) %>% 
  filter(date > '1960-01-01')  %>% 
  filter(
    !(date > '1998-12-01' & country %in% c('FRA', 'ITA', 'DEU'))
  ) %>% 
  mutate(
    
    taylor_original = 2 + cpi_core + 0.5 * (cpi_core - 2) + 0.5 * gap_hamilton,,
    
  ) %>%
  group_modify(~ {
    # Estimate linear rule for each country
    est_sample <- .x %>%
      filter(date >= as.Date("1979-01-01"), date <= as.Date("2008-12-01"))
    
    fit <- lm(policy_rate ~ cpi_core + gap_hamilton, data = est_sample)
    
    constant <-  coef(fit)["(Intercept)"] 
    cpi_coeff <- coef(fit)["cpi_core"] 
    gap_coeff <- coef(fit)["gap_hamilton"] 
    
    # Compute the fitted rule
    .x %>% mutate(taylor_best_fit = constant + cpi_coeff * cpi_core + gap_coeff * gap_hamilton)
  }) %>% 
  ungroup() %>% 
  pivot_longer(
    cols = c(policy_rate, starts_with('taylor'))
  ) %>% 
  drop_na(value) %>% 
  ggplot(aes(date, value, color = name, linetype = name)) +
  geom_line() +
  facet_wrap(~country_name, ncol = 2, scales = 'fixed') +
  coord_cartesian(ylim = c(-5, 20)) +
  scale_linetype_manual(
    values = c("solid","dashed","solid"),
    labels = c('Policy Rate', 'Best Linear Fit', 'Taylor Rule')
  ) +
  scale_color_manual(values = c("#355070", "#E56B6F", "#E56B6F"),
                     labels = c('Policy Rate', 'Best Linear Fit', 'Taylor Rule')) +
  theme(axis.title.y = element_blank())
ggsave('Output/fig9.pdf', height = 7, width = 6.5)



###############################################################################%
#### Table B1 - Estimated Taylor Rules for G7 ####
###############################################################################%

table_b1 <- df %>% 
  filter(
    !(date > '2000-01-01' & country %in% c('FRA', 'ITA', 'DEU'))
  ) %>% 
  group_by(country) %>% 
  group_modify(~ {
    
    est_sample <- .x %>%
      filter(date >= as.Date("1979-01-01"), date <= as.Date("2008-12-01"))
    
    fit <- lm(policy_rate ~ cpi_core + gap_hamilton, data = est_sample)
    
    tibble(
      constant   = coef(fit)["(Intercept)"],
      cpi_coeff  = coef(fit)["cpi_core"],
      gap_coeff  = coef(fit)["gap_hamilton"] 
    )
  }) %>%
  ungroup()

sink("Output/tableB1.tex")
table_b1 %>% 
  mutate(across(where(is.numeric), ~ round(.x, 2))) %>%
  kable(format = "latex", booktabs = TRUE, 
        linesep = "", digits = 2,
        caption = "Estimated Taylor Rules for G7",
        col.names = c("Country", "Constant", "Inflation Coeff.",
                      "Output Gap Coeff.")) 
sink()



###############################################################################%
#### Figure B1 - More Simple Rules for G7 Countries ####
###############################################################################%
df %>% 
  group_by(country) %>% 
  mutate(
    taylor_original = 2 + cpi_core + 0.5 * (cpi_core - 2) + 0.5 * gap_hamilton,
    lag_policy = lag(policy_rate, 1)
  ) %>%
  drop_na(cpi, gap_hamilton, lag_policy) %>%
  group_modify(~ {
    est_sample <- .x %>%
      filter(date >= as.Date("1979-01-01"), date <= as.Date("2008-12-01"))
    
    # Estimate linear rule for each country, with partial adjustment
    fit <- lm(policy_rate ~ cpi_core + gap_hamilton + lag_policy, data = est_sample)
    
    # Compute the long run coefficients
    constant <-  coef(fit)["(Intercept)"] / (1 - coef(fit)["lag_policy"])
    cpi_coeff <- coef(fit)["cpi_core"] / (1 - coef(fit)["lag_policy"])
    gap_coeff <- coef(fit)["gap_hamilton"] / (1 - coef(fit)["lag_policy"])
    
    # Compute the fitted rule
    .x %>% mutate(taylor_cgg = constant + cpi_coeff * cpi_core + gap_coeff * gap_hamilton)
  }) %>%
  ungroup() %>% 
  pivot_longer(
    cols = c(policy_rate, starts_with('taylor'))
  ) %>% 
  drop_na(value) %>% 
  filter(date > '1960-01-01')  %>% 
  filter(
    !(date > '1998-12-01' & country %in% c('FRA', 'ITA', 'DEU'))
  ) %>% 
  ggplot(aes(date, value, color = name, linetype = name)) +
  geom_line() +
  facet_wrap(~country_name, ncol = 2, scales = 'fixed') +
  coord_cartesian(ylim = c(-5, 20)) +
  scale_linetype_manual(
    values = c("solid","dashed","solid"),
    labels = c('Policy Rate', 'Estimated Rule', 'Taylor Rule')
  ) +
  scale_color_manual(values = c("#355070", "#E56B6F", "#E56B6F"),
                     labels = c('Policy Rate', 'Estimated Rule', 'Taylor Rule')) +
  theme(axis.title.y = element_blank())

ggsave('Output/figB1.pdf', height = 7, width = 6.5)


###############################################################################%
#### Table B2 ####
###############################################################################%
table_b2 <- df %>% 
  group_by(country) %>% 
  mutate(
    lag_policy = lag(policy_rate, 1)
  ) %>%
  drop_na(cpi, gap_hamilton, lag_policy) %>%
  group_modify(~ {
    est_sample <- .x %>%
      filter(date >= as.Date("1979-01-01"), date <= as.Date("2008-12-01"))
    
    fit <- lm(policy_rate ~ cpi_core + gap_hamilton + lag_policy, data = est_sample)
    
    tibble(
      constant   = coef(fit)["(Intercept)"],
      cpi_coeff  = coef(fit)["cpi_core"],
      gap_coeff_lr  = coef(fit)["gap_hamilton"],
      persistence_coeff = coef(fit)["lag_policy"]
    )
  }) %>%
  ungroup()

sink("Output/tableB2.tex")
table_b2 %>% 
  mutate(across(where(is.numeric), ~ round(.x, 2))) %>%
  kable(format = "latex", booktabs = TRUE, 
        linesep = "", digits = 2,
        caption = "Estimated Policy Rule Coefficients (1979–2008)",
        col.names = c("Country", "Constant", "Inflation Coeff.",
                      "Output Gap Coeff.", "Lagged Rate Coeff.")) 
sink()
