###############################################################################%
# This code is to generate the early vs late risers analysis figures
###############################################################################%

### ======= Load data ======= ###
rm(list = ls())
df <- read_csv('Data/data_early_risers.csv')

###############################################################################%
#### Figure 10 ####
###############################################################################%

# Define line colors + types for each country
line_colors <- c(
  "Brazil" = "#355070",
  "Chile" = "#355070",
  "Hungary" = "#355070",
  "Poland" = "#355070",
  "United States" = "#E56B6F",
  "Euro area" = "#E56B6F",
  "Japan" = "#E56B6F",
  "Mexico" = '#E56B6F'
)

line_types <- c(
  "Brazil" = "solid",
  "Chile" = "dashed",
  "Hungary" = "twodash",
  "Poland" = "dotted",
  "Mexico" = 'twodash',
  "United States" = "solid",
  "Euro area" = "dashed",
  "Japan" = "dotted"
)


df %>%
  filter(country_name %in% names(line_colors)) %>%
  filter(date >= "2020-06-01", date <= "2024-01-01") %>%
  drop_na(policy_rate) %>%
  group_by(country) %>%
  # Difference with policy rate in 2020-06-01
  mutate(policy_rate = policy_rate - first(policy_rate)) %>%
  ungroup() %>%
  ggplot(aes(x = date, y = policy_rate, color = country_name, linetype = country_name)) +
  geom_line(size = 0.6) +
  scale_color_manual(values = line_colors) +
  scale_linetype_manual(values = line_types) +
  labs(y = "Policy Rate", color = "Country", linetype = "Country") +
  theme(
    axis.title.y = element_blank(),
    legend.position = c(0.45, 0.98), 
    legend.margin = margin(t = 2, r = 4, b = 2, l = 4),  # padding inside the legend box
    legend.justification = c("right", "top"),  # align anchor to top-right
    legend.background = element_rect(fill = "white", color = "grey80"),
    legend.title = element_blank()
  ) +
  guides(color = guide_legend(nrow = 4)) 

ggsave('Output/fig10.pdf', height = 4, width = 4.9)



###############################################################################%
#### Figure 11 - Policy Rate and Inflation for Early and Late Risers ####
###############################################################################%

policy_change_df <- df %>%
  filter(date %in% as.Date(
    c("2021-01-01", "2022-07-01", "2024-01-01"))) %>%
  select(country, date, policy_rate) %>%
  pivot_wider(
    names_from = date,
    values_from = policy_rate,
    names_prefix = "rate_") %>%
  mutate(
    # Compute the changes between Jan 2021 to July 2022 and July 2022 to Jan 2024
    policy_rate_early = `rate_2022-07-01` - `rate_2021-01-01`,
    policy_rate_late = `rate_2024-01-01` - `rate_2022-07-01`,
    category = case_when(
      
      # Classify countries based on that changes
      policy_rate_early > policy_rate_late  ~ 'early',
      
      policy_rate_late > policy_rate_early ~ 'late'
      
      
    )
  ) %>% 
  select(country, category)



late_rate <- df %>% 
  left_join(policy_change_df) %>% 
  filter(category == 'late') %>% 
  filter(
    date >= "2021-01-01",
    date <= "2024-01-01"
  ) %>% 
  drop_na(policy_rate) %>% 
  # Normalize policy rate value 
  group_by(country) %>% 
  mutate(
    policy_rate =  policy_rate -  policy_rate[1] 
  ) %>% 
  ungroup() %>% 
  group_by(date) %>% 
  mutate(late_average = mean(policy_rate)) %>% 
  ungroup() %>% 
  ggplot() +
  geom_line(aes(date, policy_rate, group = country), color = "lightgrey", show.legend = FALSE) +
  geom_line(aes(date, late_average, color = "Average Change in Policy Rate of Late Risers"), size = 1.2) +
  scale_color_manual(values = c("Average Change in Policy Rate of Late Risers" = "#355070")) +
  scale_y_continuous(
    limits = c(-0.5, 13),
    sec.axis = dup_axis(name = "Policy Rate")  
  ) +
  theme(
    legend.position = "none",
    axis.title.y = element_blank(),    
    axis.title.y.right = element_text(size = 11),
    text = element_text(size=11),
    plot.title = element_text(hjust = 0.5)
  ) +
  labs(
    y='Policy Rate',
    title = 'Late Risers'
  ) 


early_rate <- df %>% 
  left_join(policy_change_df) %>% 
  filter(category == 'early') %>% 
  filter(
    date >= "2021-01-01",
    date <= "2024-01-01"
  ) %>% 
  drop_na(policy_rate) %>% 
  # Normalize policy rate value 
  group_by(country) %>% 
  mutate(
    policy_rate =  policy_rate -  policy_rate[1] 
  ) %>% 
  ungroup() %>% 
  group_by(date) %>% 
  mutate(late_average = mean(policy_rate)) %>% 
  ungroup() %>% 
  ggplot() +
  geom_line(aes(date, policy_rate, group = country), color = "lightgrey", show.legend = FALSE) +
  geom_line(aes(date, late_average, color = "Average Change in Policy Rate of Early Risers"), size = 1.2) +
  scale_color_manual(values = c("Average Change in Policy Rate of Early Risers" = "#355070")) +
  ylim(-0.5, 13) +
  theme(legend.position = "none",
        text = element_text(size=11),
        plot.title = element_text(hjust = 0.5)) +
  labs(
    y='Policy Rate',
    title = 'Early Risers'
  ) 



late_inflation <- df %>% 
  left_join(policy_change_df) %>% 
  filter(category == 'late') %>% 
  filter(
    date >= "2021-01-01",
    date <= "2024-01-01"
  ) %>% 
  drop_na(policy_rate) %>% 
  # Normalize policy rate value 
  group_by(country) %>% 
  ungroup() %>% 
  group_by(date) %>% 
  mutate(late_average = mean(cpi)) %>% 
  ungroup() %>% 
  ggplot() +
  geom_line(aes(date, cpi, group = country), color = "lightgrey", show.legend = FALSE) +
  geom_line(aes(date, late_average, color = "Average Inflation of Late Risers"), size = 1.2) +
  scale_color_manual(values = c("Average Inflation of Late Risers" = "#355070")) +
  scale_y_continuous(
    limits = c(-0.5, 26),
    sec.axis = dup_axis(name = "Inflation")  
  ) +
  theme(legend.position = "none",
        axis.title.y = element_blank(),    
        axis.title.y.right = element_text(size = 11),
        text = element_text(size=11)
  ) +
  labs(y='Inflation') 


early_inflation <- df %>% 
  left_join(policy_change_df) %>% 
  filter(category == 'early') %>% 
  filter(
    date >= "2021-01-01",
    date <= "2024-01-01"
  ) %>% 
  drop_na(policy_rate) %>% 
  # Normalize policy rate value 
  group_by(country) %>% 
  ungroup() %>% 
  group_by(date) %>% 
  mutate(late_average = mean(cpi)) %>% 
  ungroup() %>% 
  ggplot() +
  geom_line(aes(date, cpi, group = country), color = "lightgrey", show.legend = FALSE) +
  geom_line(aes(date, late_average, color = "Average Inflation of Early Risers"), size = 1.2) +
  scale_color_manual(values = c("Average Inflation of Early Risers" = "#355070")) +
  ylim(-0.5, 26) +
  theme(legend.position = "none",
        text = element_text(size=11)) +
  labs(y='Inflation') 

cowplot::plot_grid(early_rate, late_rate,
                   early_inflation, late_inflation,
                   ncol=2)

ggsave('Output/fig11.pdf', height = 5.5, width = 6)



###############################################################################%
#### Figure 12 - Early Rate Hikes and Historical Inflation ####
###############################################################################%

# Compute average inflation history
average_inflation <- df %>% 
  filter(
    date >= '1990-01-01',
    date <= '2019-12-01'
  ) %>% 
  group_by(country) %>% 
  summarise(
    average_inflation = mean(cpi)
  ) %>% 
  ungroup() 

# Compute rate change between Jan 2021 to July 2022
magnitude_raise <- df %>% 
  filter(
    date %in% c('2021-01-01', '2022-07-01')
  ) %>% 
  pivot_wider(
    id_cols = country,
    names_from = 'date',
    values_from = 'policy_rate'
  ) %>% 
  mutate(
    magnitude_raise = `2022-07-01` - `2021-01-01`
  ) %>% 
  select(country, magnitude_raise) 

full_join(magnitude_raise, average_inflation) %>%
  mutate(log_avg_infl = log(average_inflation)) %>%
  ggplot(aes(x = log_avg_infl, y = magnitude_raise)) +
  geom_point(color = "#355070") +
  geom_smooth(method = 'lm', se = FALSE, color = "#355070") +
  geom_text_repel(aes(label = country), size = 3) +
  scale_x_continuous(
    breaks = log(c(1, 2, 5, 10, 50, 200)),          
    labels = c("1", "2", "5", "10", "50", "200")
  ) +
  labs(
    y = 'Change in Policy Rate Jan 2021 - Jul 2022',
    x = 'Average Inflation 1990–2019'
  ) +
  theme(axis.title.x = element_text(size = 10),
        axis.title.y = element_text(size = 10))
ggsave('Output/fig12.pdf', height = 3, width = 3)

