clear all; close all;

% This code produces figures 13 and 16

%% Figure 13 - Determinacy in a Less Forward-Looking Model

% Parameters
p.beta = 0.99;
p.sigma = 1;
p.kappa = 0.024;

% Grids for omega and phi

npoints   = 500;    
omega  = linspace(0,1,npoints);
phi    = linspace(0,2,npoints);
[OMEGA,PHI]  = meshgrid(omega,phi);

% Compute the determinacy condition

det_region = (p.beta .* OMEGA .* (1-OMEGA) < (1-OMEGA) + p.sigma*p.kappa*(PHI-OMEGA) );

% Aesthetics 

f = figure;
f.Theme = "light";
f.Units = 'centimeters';
f.Position = [5, 5, 14, 12];

imagesc(omega, phi, det_region);
set(gca,'YDir','normal');      
xlabel('\omega','FontSize',12);
ylabel('\phi','FontSize',12);


cm = [0.7 0.7 0.7; 
      1.0 1.0 1.0]; 
colormap(cm);
caxis([0 1]);             
hold on;

hDet = plot(NaN, NaN, 's', ...
            'MarkerFaceColor', [1 1 1], ...
            'MarkerEdgeColor', 'k', ...
            'MarkerSize', 12);
hInd = plot(NaN, NaN, 's', ...
            'MarkerFaceColor', [0.7 0.7 0.7], ...
            'MarkerEdgeColor', 'k', ...
            'MarkerSize', 12);
legend([hDet hInd], {'Determinacy','Indeterminacy'},'Location', 'northeast');
hold off 

print(gcf, '../Output/fig13', '-dpng', '-r300');



%% Figure 16 - Taylor Principle with Increasing Relative Variance of Cost-Push Shocks
clear; 

% Parameters
p.beta = 0.99;
p.sigma = 1;
p.kappa = 0.024;
p.rho_S = 0.85;
p.rho_D = 0.85;

p.alpha = 1;
rho_pos = 1;
rho_neg = -1;

tsim = 1e5; 
burn = 100;

% Define grids for relative variance
N = 100;  
VD = 1; % fix the variance of demand shocks
VS_grid = logspace(log10(0.05), log10(200), N);

[phi_neg, phi_pos] = deal(zeros(N, 1));


% Simulate and run regressions
for j = 1:N

    % Positive Correlation
    sim_commit_pos = sim_optimal_commit(p, VS_grid(j), VD, rho_pos, tsim, burn);

    % Regression
    X = sim_commit_pos(:, 1:2);
    X = [ones(size(X, 1), 1), X];
    y = sim_commit_pos(:,4);  
    b_opt_commit = regress(y, X);
    phi_y_pos   = b_opt_commit(2); % Output gap is second coefficient 
    phi_pi_pos  = b_opt_commit(3); % Inflation is third coefficient

    phi_pos(j) = phi_pi_pos + phi_y_pos * (1 - p.beta) / (4*p.kappa);

    % Negative Correlation
    sim_commit_neg = sim_optimal_commit(p, VS_grid(j), VD, rho_neg, tsim, burn);

    % Regression
    X = sim_commit_neg(:, 1:2);
    X = [ones(size(X, 1), 1), X];
    y = sim_commit_neg(:,4);  
    b_opt_commit = regress(y, X);
    phi_y_neg  = b_opt_commit(2); % Output gap is second coefficient 
    phi_pi_neg = b_opt_commit(3); % Inflation is third coefficient

    phi_neg(j) = phi_pi_neg + phi_y_neg * (1 - p.beta)/(4*p.kappa);


end

% Zero Correlation: run it with VD = 0 since it does not affect the regression
sim_commit = sim_optimal_commit(p, 1, 0, 0, tsim, burn);

% Regression
X = sim_commit(:, 1:2);
X = [ones(size(X, 1), 1), X];
y = sim_commit(:,4);  
b_opt_commit = regress(y, X);
phi_y_neg      = b_opt_commit(2); % Output gap is second coefficient 
phi_pi_neg = b_opt_commit(3); % Inflation is third coefficient

phi = phi_pi_neg + phi_y_neg * (1 - p.beta) / (4*p.kappa);


% Plot the figure 
x_axis = VS_grid ./ (VD + VS_grid);

f = figure;
f.Theme = "light";
plot(x_axis, phi_neg,  'Color', '#355070', 'LineWidth', 2, 'LineStyle', '-');
hold on
plot(x_axis, phi_pos,  'Color', '#355070', 'LineWidth', 2, 'LineStyle', ':');
yline(phi, 'Color', '#355070', 'LineWidth', 2, 'LineStyle', '--'); 
hold off
ylim([0, 1.5])
legend({'Negatively Correlated Shocks', 'Positively Correlated Shocks', 'Uncorrelated Shocks'}, 'Location', 'Northeast');
ylabel('$\phi_{\pi} + \frac{1-\beta}{4\kappa} \phi_y$', 'Interpreter', 'latex');
xlabel('Share of Cost Push Shocks in Total Shock Variance');
set(gca, 'Position', [0.11, 0.11, 0.85, 0.85]); 
set(gca, 'FontSize', 12);
print(gcf, '../Output/fig16', '-dpng', '-r300');

