clear; close all;


%% Parameters

p.beta = 0.99;
p.sigma = 1;
p.kappa = 0.024;
p.rho_S = 0.85;
p.rho_D = 0.85;

tsim = 1e5; 
burn = 100;

alpha_vec = [10, 5, 2, 1, 0.5, 0.2, 0.1, 0.05, 0.02, 0.01];


%% Table 1 - Estimated Taylor Rules under Optimal Policy with Commitment
[phi_pi, phi_y] = deal(zeros(length(alpha_vec), 1));

for j = 1 : length(alpha_vec)

    p.alpha = alpha_vec(j);

    % Run Simulation
    sim_commit = sim_optimal_commit(p, 1, 0, 0, tsim, burn); % variance of supply shock = 1, variance demand shock = 0, shock correlation = 0

    % Regression
    X = sim_commit(:, 1:2);
    X = [ones(size(X, 1), 1), X];
    nom_rate = sim_commit(:,4);  
    b_opt_commit = regress(nom_rate, X);
    phi_y(j)      = b_opt_commit(2); % Output gap is second coefficient 
    phi_pi(j) = b_opt_commit(3); % Inflation is third coefficient


end

phi_aug = phi_pi + phi_y * (1 - p.beta) / (4*p.kappa);

% Print the table

fprintf('Table 1 - Estimated Taylor Rules under Optimal Policy with Commitment\\\\\n')
fprintf('$\\alpha$ & Inflation & Output Gap & Combined \\\\\n');
for i = 1:length(alpha_vec)
    fprintf('%.2f & %.2f & %.2f & %.2f \\\\\n', round(alpha_vec(i),2), round(phi_pi(i),2), ...
        round(phi_y(i),2), round(phi_aug(i), 2));
end


% Export the table

fileID = fopen('../Output/table1.tex', 'w');

fprintf(fileID, '\\begin{tabular}{c c c c}\n');
fprintf(fileID, '\\toprule\n');
fprintf(fileID, '$\\alpha$ & Inflation & Output Gap & Combined \\\\\n');
fprintf(fileID, '\\midrule\n');
fprintf(fileID, '\\midrule\n');

for i = 1:length(alpha_vec)
    fprintf(fileID, '%.2f & %.2f & %.2f & %.2f \\\\\n', round(alpha_vec(i),2), round(phi_pi(i),2), ...
        round(phi_y(i),2), round(phi_aug(i), 2));
end

fprintf(fileID, '\\bottomrule\n');
fprintf(fileID, '\\end{tabular}\n');

fclose(fileID);


%% Table 2 - Estimated Taylor Rule with Optimal Policy under Commitment and Correlated Shocks

VS = 1;
VD = 1;
rho_pos = 1;
rho_neg = -1;

[phi_pi_pos, phi_y_pos] = deal(zeros(length(alpha_vec), 1));
[phi_pi_neg, phi_y_neg] = deal(zeros(length(alpha_vec), 1));

for j = 1 : length(alpha_vec)

    p.alpha = alpha_vec(j);

    % Positive Correlation
    sim_commit = sim_optimal_commit(p, VS, VD, rho_pos, tsim, burn);

    X = sim_commit(:, 1:2);
    X = [ones(size(X, 1), 1), X];
    nom_rate = sim_commit(:,4);  
    b_opt_commit  = regress(nom_rate, X);
    phi_y_pos(j)  = b_opt_commit(2); % Output gap is second coefficient 
    phi_pi_pos(j) = b_opt_commit(3); % Inflation is third coefficient

    % Negative Correlation
    sim_commit = sim_optimal_commit(p, VS, VD, rho_neg, tsim, burn);

    X = sim_commit(:, 1:2);
    X = [ones(size(X, 1), 1), X];
    nom_rate = sim_commit(:,4);  
    b_opt_commit  = regress(nom_rate, X);
    phi_y_neg(j)  = b_opt_commit(2); % Output gap is second coefficient 
    phi_pi_neg(j) = b_opt_commit(3); % Inflation is third coefficient


end

phi_aug_pos = phi_pi_pos + phi_y_pos * (1 - p.beta) / (4 * p.kappa);
phi_aug_neg = phi_pi_neg + phi_y_neg * (1 - p.beta) / (4 * p.kappa);

% Print the table
fprintf('Table 2 - Estimated Taylor Rule with Optimal Policy under Commitment and Correlated Shocks \\\\\n')
fprintf('& \\multicolumn{3}{c}{$\\rho = -1$} & & \\multicolumn{3}{c}{$\\rho = 1$} \\\\\n');
fprintf('$\\alpha$ & Inflation & Output Gap & $\\phi_{\\pi} + \\phi_y \\frac{1-\\beta}{4\\kappa}$ & & Inflation & Output Gap & $\\phi_{\\pi} + \\phi_y \\frac{1-\\beta}{4\\kappa}$ \\\\\n');
for i = 1:length(alpha_vec)
    fprintf('%.2f & %.2f & %.2f & %.2f & & %.2f & %.2f & %.2f \\\\\n', ...
        round(alpha_vec(i), 2), ...
        round(phi_pi_neg(i), 2), round(phi_y_neg(i), 2), round(phi_aug_neg(i), 2), ...
        round(phi_pi_pos(i), 2), round(phi_y_pos(i), 2), round(phi_aug_pos(i), 2));
end

% Export the table

fileID = fopen('../Output/table2.tex', 'w');

fprintf(fileID, '\\begin{tabular}{c c c c c c c c}\n');
fprintf(fileID, '\\toprule\n');
fprintf(fileID, '& \\multicolumn{3}{c}{$\\rho = -1$} & & \\multicolumn{3}{c}{$\\rho = 1$} \\\\\n');
fprintf(fileID, '\\cmidrule(lr){2-4} \\cmidrule(lr){6-8}\n');
fprintf(fileID, '$\\alpha$ & Inflation & Output Gap & $\\phi_{\\pi} + \\phi_y \\frac{1-\\beta}{4\\kappa}$ & & Inflation & Output Gap & $\\phi_{\\pi} + \\phi_y \\frac{1-\\beta}{4\\kappa}$ \\\\\n');
fprintf(fileID, '\\midrule\n');

for i = 1:length(alpha_vec)
    fprintf(fileID, '%.2f & %.2f & %.2f & %.2f & & %.2f & %.2f & %.2f \\\\\n', ...
        round(alpha_vec(i), 2), ...
        round(phi_pi_neg(i), 2), round(phi_y_neg(i), 2), round(phi_aug_neg(i), 2), ...
        round(phi_pi_pos(i), 2), round(phi_y_pos(i), 2), round(phi_aug_pos(i), 2));
end

fprintf(fileID, '\\bottomrule\n');
fprintf(fileID, '\\end{tabular}\n');

fclose(fileID);

%% Table D3 - Taylor Rules under Optimal Policy with Commitment with Flatter Phillips Curve

[phi_pi, phi_y] = deal(zeros(length(alpha_vec), 1));

p.kappa = 0.0069; % Flatter Phillips Curve

for j = 1 : length(alpha_vec)

    p.alpha = alpha_vec(j);

    % Run Simulation
    sim_commit = sim_optimal_commit(p, 1, 0, 0, tsim, burn); % variance of supply shock = 1, variance demand shock = 0, shock correlation = 0

    % Regression
    X = sim_commit(:, 1:2);
    X = [ones(size(X, 1), 1), X];
    nom_rate = sim_commit(:,4);  
    b_opt_commit = regress(nom_rate, X);
    phi_y(j)  = b_opt_commit(2); % Output gap is second coefficient 
    phi_pi(j) = b_opt_commit(3); % Inflation is third coefficient


end

phi_aug = phi_pi + phi_y * (1 - p.beta) / (4*p.kappa);


% Print the table
fprintf('Table D3 - Taylor Rules under Optimal Policy with Commitment with Flatter Phillips Curve \\\\\n')
fprintf('$\\alpha$ & Inflation & Output Gap & Combined \\\\\n');
for i = 1:length(alpha_vec)
    fprintf('%.2f & %.2f & %.2f & %.2f \\\\\n', round(alpha_vec(i),2), round(phi_pi(i),2), ...
        round(phi_y(i),2), round(phi_aug(i), 2));
end

% Export the table

fileID = fopen('../Output/tableD3.tex', 'w');

fprintf(fileID, '\\begin{tabular}{c c c c}\n');
fprintf(fileID, '\\toprule\n');
fprintf(fileID, '$\\alpha$ & Inflation & Output Gap & Combined \\\\\n');
fprintf(fileID, '\\midrule\n');
fprintf(fileID, '\\midrule\n');

for i = 1:length(alpha_vec)
    fprintf(fileID, '%.2f & %.2f & %.2f & %.2f \\\\\n', round(alpha_vec(i),2), round(phi_pi(i),2), ...
        round(phi_y(i),2), round(phi_aug(i), 2));
end

fprintf(fileID, '\\bottomrule\n');
fprintf(fileID, '\\end{tabular}\n');

fclose(fileID);


%% Table D4 - Taylor Rules under Optimal Policy with Commitment using 12-month Inflation

p.kappa = 0.024; % Adjust back the Phillips Curve

[phi_pi, phi_y] = deal(zeros(length(alpha_vec), 1));

for j = 1 : length(alpha_vec)

    p.alpha = alpha_vec(j);

    % Run Simultion
    sim_commit = sim_optimal_commit(p, 1, 0, 0, tsim, burn); % variance of supply shock = 1, variance demand shock = 0, shock correlation = 0

    output_gap =  sim_commit(:, 1);
    inflation = sim_commit(:, 2);
    nom_rate = sim_commit(:,4);  

    % Get 12 month inflation
    T = length(inflation);
    inflation_12 = NaN(T, 1);  
    
    for t = 4:T
        gross_quarterly = 1 + inflation(t-3:t) / 100;     % Take last 4 quarters
        inflation_12(t) = prod(gross_quarterly)^(1/4) - 1; 
    end
    inflation_12 = 100 * inflation_12;

    % Regression
    X = [ones(size(inflation_12, 1), 1), output_gap, inflation_12];
    b_opt_commit = regress(nom_rate, X);
    phi_y(j)  = b_opt_commit(2); % Output gap is second coefficient 
    phi_pi(j) = b_opt_commit(3); % Inflation is third coefficient


end

phi_aug = phi_pi + phi_y * (1 - p.beta) / (4*p.kappa);


% Print the table
fprintf('Table D4 - Taylor Rules under Optimal Policy with Commitment using 12-month Inflation')
fprintf('$\\alpha$ & Inflation & Output Gap & Combined \\\\\n');
for i = 1:length(alpha_vec)
    fprintf('%.2f & %.2f & %.2f & %.2f \\\\\n', round(alpha_vec(i),2), round(phi_pi(i),2), ...
        round(phi_y(i),2), round(phi_aug(i), 2));
end

% Export the table
fileID = fopen('../Output/tableD4.tex', 'w');

fprintf(fileID, '\\begin{tabular}{c c c c}\n');
fprintf(fileID, '\\toprule\n');
fprintf(fileID, '$\\alpha$ & Inflation & Output Gap & Combined \\\\\n');
fprintf(fileID, '\\midrule\n');
fprintf(fileID, '\\midrule\n');

for i = 1:length(alpha_vec)
    fprintf(fileID, '%.2f & %.2f & %.2f & %.2f \\\\\n', round(alpha_vec(i),2), round(phi_pi(i),2), ...
        round(phi_y(i),2), round(phi_aug(i), 2));
end

fprintf(fileID, '\\bottomrule\n');
fprintf(fileID, '\\end{tabular}\n');

fclose(fileID);
